/*
 * DataUtil.java
 *
 * Created on June 14, 2006, 9:19 PM
 */
package ags.communication;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.locks.LockSupport;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Collection of miscellaneous text formatting and other routines used by the host utilities.
 * @author Administrator
 */
public class DataUtil {

    public static long CPU_SPEED        = 1020500L;
    public static long NANOS_PER_SECOND = 1000000000L;
    // How long it takes to send 1 character at 115200 baud in nanoseconds
    public static long NANOS_PER_CHAR = nanosPerCharAtSpeed(115200L);
    static long nanosPerCharAtSpeed(long speed) {
        long result = (NANOS_PER_SECOND * 9L) / speed;
//        System.out.println("calculate "+result+" nanos per character at "+speed+" baud");
        return result;
    }
    static long cyclesToNanos(long cycles) {
        return (NANOS_PER_SECOND * cycles)/CPU_SPEED;

    }

    static void nanosleep(long duration) {
        if (duration > 0)
            LockSupport.parkNanos(duration);
    }

    /** Creates a new instance of DataUtil */
    private DataUtil() {
    }

    /**
     * Strip off apple high-order text
     * @param in text to strip
     * @return normalized ascii text (hopefully)
     */
    public static String convertFromAppleText(String in) {
        if (in == null) {
            return "";
        }
        StringBuffer out = new StringBuffer();
        for (char c : in.toCharArray()) {
            out.append((char) (c & 0x7f));
        }
        return out.toString();
    }

    /**
     * Convert an array of bytes (assuming 8-bit ascii) to a string
     * @param data array of bytes to convert
     * @return new string
     */
    public static String bytesToString(byte[] data) {
        if (data == null) {
            return "";
        }
        StringBuffer out = new StringBuffer();
        for (byte b : data) {
            out.append((char) b);
        }
        return out.toString();
    }

    /**
     * Convert a string of data to hexidecimal equivilants.  Useful for writing text directly to the screen.
     * @param in text
     * @return string of hex digits ready to be typed to the apple
     */
    public static String asAppleScreenHex(String in) {
        StringBuffer out = new StringBuffer();
        for (int i = 0; i < in.length(); i++) {
            char c = in.charAt(i);
            if (c >= ' ') {
                out.append(Integer.toHexString(c | 0x80)).append(' ');
            }
        }

        return out.toString();
    }

    /**
     * Given an int, break it down to a little-endian word (byte array format)
     * @param i int to convert
     * @return little endian byte array
     */
    public static byte[] getWord(int i) {
        byte word[] = new byte[2];
        word[0] = (byte) (i % 256);
        word[1] = (byte) (i / 256);
        return word;
    }

    /**
     * Open a file and get its contents as one big string
     * @param file name of file to open
     * @return The file contents as a string
     */
    public static String getFileAsString(String file) {
        InputStream stream = DataUtil.class.getResourceAsStream(file);
        StringBuffer data = new StringBuffer();
        byte buf[] = new byte[256];
        int read = 0;
        try {
            while ((read = stream.read(buf)) > 0) {
                String s = new String(buf, 0, read);
                data.append(s);
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        }
        return data.toString();
    }

    /**
     * Open a file and get its contents as one big byte array
     * @param file name of file to open
     * @throws java.io.IOException if the file or com port cannot be accessed
     * @return byte array containing file's contents
     */
    public static byte[] getFileAsBytes(String file) throws IOException {
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        int size = 0;
        try {
            size = stream.available();
        } catch (Throwable ex) {
            System.out.println("Error reading file "+file);
            Logger.getLogger(DataUtil.class.getName()).log(Level.SEVERE, null, ex);
            throw new IOException("Error reading "+file);
        }
        ByteBuffer bb = ByteBuffer.allocate(size);
        byte[] buf = new byte[256];
        int read = 0;
        try {
            while ((read = stream.read(buf)) > 0) {
                bb.put(buf, 0, read);
            }
        } catch (IOException ex) {
            Logger.getLogger(DataUtil.class.getName()).log(Level.WARNING, null, ex);
        }
        System.out.println("Read file "+file);
        return bb.array();
    }

    /**
     * Does this byte buffer contain the same values as the array of bytes?
     * @param bb buffer to check
     * @param data bytes to look for
     * @return true if bytes were found in the buffer
     */
    public static boolean bufferContains(ByteBuffer bb, byte data[]) {
        int d = 0;
        boolean match = false;
        for (int i = 0; i < bb.position() && d < data.length; i++) {
            if (bb.get(i) == data[d]) {
                match = true;
                d++;
            } else {
                match = false;
                d = 0;
            }
        }

        return match;
    }

    /**
     * Sleep the current thread for a little while
     * @param time time in ms to wait
     */
    public static void wait(int time) {
        try {
            Thread.sleep(time);
        } catch (InterruptedException ex) {
            ex.printStackTrace();
        }
    }

    public static byte[] packbits(int target, byte[] input) {
        List<Byte> out = new ArrayList<Byte>();
        out.add((byte) (0x0ff & target));
        out.add((byte) ((0x0ff00 & target) >> 8));
        int pos = 0;
        while (pos < input.length) {
            boolean rawData = true;
            if (pos < input.length - 4) {
                if (input[pos] == input[pos+2] && input[pos+1] == input[pos+3]) {
                    rawData = false;
                    int numberReps=0;
                    int seek = pos + 4;
                    boolean foundPattern = true;
                    // Keep going until:
                    // 1) Reach end of data
                    // 2) Stop finding matches
                    // 3) Get 129 total pairs (first two + numberReps)
                    while (seek < input.length-2 && foundPattern && numberReps < 127) {
                        if (input[pos] == input[seek] && input[pos+1] == input[seek+1]) {
                            numberReps++;
                            seek += 2;
                        } else {
                            foundPattern = false;
                        }
                    }
                    byte size = (byte) (0x0ff & (128 + numberReps));
                    out.add(size);
                    out.add(input[pos]);
                    out.add(input[pos+1]);
                    pos = seek;
                }
            }
            // No pattern, just output raw data until
            // 1) We hit a repeating pattern
            // 2) We hit end of data
            // 3) We hit 127 characters
            if (rawData) {
                int seek = pos;
                boolean foundPattern = false;
                int count = 0;
                while (seek < input.length && !foundPattern && count < 127) {
                    if (seek < input.length - 4 && input[seek] == input[seek+2] && input[seek+1] == input[seek+3]) {
                        foundPattern = true;
                    } else {
                        seek++;
                        count++;
                    }
                }
                out.add((byte) (0x0ff & count));
                for (int i=0; i < count; i++) {
                    out.add(input[pos+i]);
                }
                pos = seek;
            }
        }
        out.add((byte) 0);

//        System.out.println("Packbits: "+input.length+" bytes compressed to "+out.size()+" bytes ("+(100-(100*out.size()/input.length))+"% compression)");

        // Convert back to native array
        byte[] result = new byte[out.size()];
        for (int i=0; i < out.size(); i++)
            result[i] = out.get(i);
        return result;
    }

    public static byte[] xor(byte[] b1, byte[] b2) {
        if (b1.length != b2.length) return null;
        byte[] out = new byte[b1.length];
        for (int i=0; i < b1.length; i++) {
            out[i] = (byte) (0x0ff & (b1[i] ^ b2[i]));
        }
        return out;
    }

    public static byte[] truncateBeginningZeros(byte[] in) {
        int lastRealData = 0;
        while (lastRealData < in.length && in[lastRealData] == 0) {
            lastRealData++;
        }
        if (lastRealData == 0) {
            return in;
        }
        byte[] result = new byte[in.length-lastRealData];
        for (int i=lastRealData; i < in.length; i++)
            result[i-lastRealData] = in[i];
        return result;
    }

    public static byte[] truncateEndingZeros(byte[] in) {
        int lastRealData = in.length-1;
        while (lastRealData >= 0 && in[lastRealData] == 0) {
            lastRealData--;
        }
        if (lastRealData == in.length-1) {
            return in;
        }
        byte[] result = new byte[lastRealData+1];
        for (int i=0; i < lastRealData+1; i++) {
            result[i] = in[i];
        }
        return result;
//        return Arrays.copyOf(in, lastRealData+1);
    }

    public static byte[] packScreenUpdate(int address, byte[] oldScreen, byte[] newScreen) {
        byte[] diff = xor(oldScreen, newScreen);
        byte[] smaller = truncateEndingZeros(diff);
        byte[] evenSmaller = truncateBeginningZeros(smaller);
        address += (smaller.length - evenSmaller.length);
        return packbits(address, evenSmaller);
    }
}